
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; EEPROM05.ASM : (c) Pierre COL, janvier 2000, Couplage PIC16F84 & 24C16 ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;**************************************************************************;

;;;; Definition et parametres (utiles a l'assembleur) : ;;;;

              LIST P=16F84, R=DEC

#include      "P16F84.INC"

              __CONFIG  _CP_OFF & _PWRTE_ON & _WDT_OFF & _XT_OSC
              __IDLOCS  1998h

;**************************************************************************;

;;;; Etiquettes : ;;;;

;;; Adresses programme :
DEBUT         equ       000h ; Debut de la memoire physique.
INT           equ       004h ; Routine d'interruption (pas utilisee).
MEM           equ       006h ; Endroit ou stocker des constantes.

;;; Adresses RAM :
DebutRAM      equ       00Ch
BOOLEEN       equ       (DebutRAM+1)    ; - 8 bits = 8 variables booleennes.
I2C_data      equ       (BOOLEEN+1)     ; - Octet de donnee.
I2C_adrHIGH   equ       (I2C_data+1)    ; - Adresse : octet de poids faible.
I2C_adrLOW    equ       (I2C_adrHIGH+1) ; - Adresse : octet de poids fort.
VAR1          equ       (I2C_adrLOW+1)  ; - 3 variables temporaires :
VAR2          equ       (VAR1+1)        ;   sauvegarde de W, compteurs, etc.
LSB           equ       (VAR2+1)   ;
MSB           equ       (LSB+1)    ;

;;; Bits du Port A :
PA0           equ       0
PA1           equ       1
PA2           equ       2
PA3           equ       3 ; (sur lequel se trouve SCL).
PA4           equ       4 ; (sur lequel se trouve SDA).

;;; Entrees / sorties :
LED           equ       PA0 ; Broche sur laquelle est connectee une Led.
SCL           equ       PA3 ; Broche 'Serial CLock' du bus I2C.
SDA           equ       PA4 ; Broche 'Serial DAta' du bus I2C.

;;; BOOLEEN :
I2CACK        equ       0 ; le bit '0' de l'adresse BOOLEEN, qui sert a
                          ; verifier si l'acces a la memoire I2C est correct.
;;; Constantes :
I2C_WR        equ       00000000b
I2C_RD        equ       00000001b
I2C_EEPROM    equ       10100000b

;**************************************************************************;

;;;; Donnees EEPROM interne du PIC : ;;;;

EEPROM_DATA   equ       2100h

              ORG       EEPROM_DATA          ; Contenu quelconque sous
              DE        000h,001h,002h,003h  ; des formes differentes...
              DE        004h,005h,006h,007h  ; Ces 64 octets d'EEPROM
              DE        008h,009h,00Ah,00Bh  ; internes ne sont pas
              DE        00Ch,00Dh,00Eh,00Fh  ; utilises pas notre
              DE        000h,010h,020h,030h  ; application...
              DE        040h,050h,060h,070h
              DE        080h,090h,0A0h,0B0h
              DE        0C0h,0D0h,0E0h,0F0h
              DE        "0123456789ABCDEF"
              DE        "FEDCBA9876543210"

;**************************************************************************;

;; Divers actions, non utiles dans ce cas precis :

              ORG       DEBUT
              GOTO      PROG

              ORG       INT    ; Retour d'interruption (mais
              RETFIE           ; celle-ci n'est pas utilisee).

              ORG       MEM    ; Contantes ou textes divers a stocker :
Message       DT        "Debut de la memoire",0,1,2,3,4,5

;; Debut du programme : ;;;;

PROG          BSF       STATUS,RP0 ; Pour acceder aux registres $80 a $AF.

              MOVLW     00000b            ; Configuration des ports A et B :
              MOVWF     TRISA & 01111111b ; - OUT: RA0=Led ; RA3/RA4=bus I2C.
              MOVLW     00000000b         ; - OUT: RA1 et RA2 non utilisees.
              MOVWF     TRISB & 01111111b ; - OUT: tout le Port B.
              MOVLW     01111111b              ; Registres d'options : B7=0
              MOVWF     OPTION_REG & 01111111b ; => res. de tirage actives.

              BCF       STATUS,RP0 ; Pour acceder aux registres $00 a $2F,
                                   ; notamment les Ports A et B.

              BSF       PORTA,LED  ; Led eteinte : [PA0]-[2k2]-(led)-[+5V].

              BSF       PORTA,SCL  ; Initialiser a '1' les
              BSF       PORTA,SDA  ; deux lignes du bus I2C.

              CLRF      BOOLEEN    ; Initialiser d'un coup a '0' les
                                   ; 8 bits de la variable booleenne ;
                                   ; en fait, seul le bit 0 est utilise.

;; Lecture de l'octet a l'adresse $0000 de l'EEPROM
;; I2C et affichage sur le Port B du PIC :

              MOVLW     00h         ; - On charge l'adresse $0000, donc on
              MOVWF     I2C_adrLOW  ; met $00 dans l'octet de poids faible
              MOVLW     00h         ; de l'adresse, et $00 dans l'octet de
              MOVWF     I2C_adrHIGH ; poids fort. on aurait pu ecrire :
                                    ; CLRF I2C_adrLOW
                                    ; CLRF I2C_adrHIGH
                                    ; pour simplifier dans ce cas precis.

              CALL      Read24C16   ; On lit la memoire a l'adresse choisie.
              MOVF      I2C_data,W  ; On recupere le contenu de I2C_data
              MOVWF     PORTB       ; dans W, et on l'envoie sur le Port B.

;; La donnee est maintenant lue et affichee sur le Port B.

;; Remplissage de la memoire de l'adresse $0001 a $07FF avec une valeur
;; qui augmente : $01, $02, $03, etc... (en fait, on prend simplement
;; l'octet de poids faible de l'adresse en tant que donnee).

              MOVLW     01h         ; - On part de l'adresse $0001
              MOVWF     LSB         ; (car l'octet numero 1 a l'adresse
              MOVLW     00h         ; $0000 n'est pas modifie, mais
              MOVWF     MSB         ; envoye sur le Port B)

NextBit       MOVF      LSB,W       ; - On charge l'octet de poids
              MOVWF     I2C_adrLOW  ; faible de l'adresse, puis
              MOVF      MSB,W       ; l'octet de poids fort, dans
              MOVWF     I2C_adrHIGH ; les registres adequats.

              MOVF      LSB,W       ; - On charge l'octet de donnee qui est
              MOVWF     I2C_data    ; aussi le poids faible de l'adresse.
              CALL      Write24C16  ; - On ecrit l'octet dans la memoire.

              INCF      LSB,F       ; On incremente l'octet de poids faible
              BTFSS     STATUS,Z    ; de l'adresse. si on ne repasse pas par
              GOTO      NextBit     ; zero, cela veut dire pas de retenue :
                                    ; on boucle pour programmer l'octet.

              INCF      MSB,F       ; - Si il y a une retenue, on incremente
              BTFSS     MSB,3       ; l'octet de poids fort puis on boucle,
              GOTO      NextBit     ; sauf si on arrive a la fin de la
                                    ; memoire, c'est a dire si le bit
                                    ; d'adresse 11 (bit 3 du MSB) passe a 1.

;; La memoire est maintenant remplie (sauf le premier octet, bien sur).

              BCF       PORTA,LED    ; Allumer la Led.
Reboucler     GOTO      Reboucler    ; On attend patiemment la fin du monde.

;; Fin du programme.


;; Autres exemples d'utilisation de Read24C16 et Write24C16 :
;;
;;            MOVLW     01h           ; On lit la donnee se
;;            MOVWF     I2C_adrHIGH   ; trouvant a l'adresse
;;            MOVLW     75h           ; $0175 ; la valeur ainsi
;;            MOVWF     I2C_adrLOW    ; obtenue est disponible
;;            CALL      Read24C16     ; dans la variable I2C_data.
;;
;;            MOVLW     27h           ; On ecrit la donnee
;;            MOVWF     I2C_data      ; fournie dans la variable
;;            MOVLW     02h           ; I2C_data (ici elle vaut
;;            MOVWF     I2C_adrHIGH   ; $27) a l'adresse $0231.
;;            MOVLW     31h
;;            MOVWF     I2C_adrLOW
;;            CALL      Write24C16
;;

;**************************************************************************;

;--------------------------------------------------------------------------;
;        Fonctions principales pour ecrire et lire dans l'EEPROM           ;
;        I2C, octet par octet (pour les modeles 24C01 a 24C16).            ;
;--------------------------------------------------------------------------;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Lit l'octet situe dans l'EEPROM I2C et met le contenu dans [I2C_data]. ;;
;; Pour indiquer l'adresse a lire :                                       ;;
;; les 3 bits de poids fort de l'adresse sont mis dans [I2C_adrHIGH].     ;;
;; les 8 bits de poids faible de l'adresse sont mis dans [I2C_adrLOW].    ;;
;; (attention, cela modifie les variables VAR1, VAR2 et BOOLEEN)          ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
Read24C16     CALL      Set_Adr_I2C         ; - Pointer l'adresse.
              CALL      I2CSTART            ; - Envoyer un ordre de
              RLF       I2C_adrHIGH,W       ;   lecture a l'EEPROM I2C.
              ANDLW     00001110b           ;
              IORLW     (I2C_EEPROM+I2C_RD) ;    (octet : 1010xxx1b
              CALL      WtoI2C              ;    xxx = MSB adresse)
              CALL      I2CgetACK           ; - Confirme par un ACK.
              CALL      I2CtoW              ; - Lire l'octet.
              MOVWF     I2C_data            ; - Le mettre dans [I2C_data]
              CALL      I2CsetNoACK         ; - Terminer l'operation
              CALL      I2CSTOP             ;   (pas d'ACK, mais un STOP).
              RETURN                        ; - C'est fini !

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Ecrit l'octet situe dans [I2C_data] dans l'EEPROM I2C.              ;;
;; Pour indiquer l'adresse ou ecrire :                                 ;;
;; les 3 bits de poids fort de l'adresse sont mis dans [I2C_adrHIGH].  ;;
;; les 8 bits de poids faible de l'adresse sont mis dans [I2C_adrLOW]. ;;
;; (attention, cela modifie les variables VAR1, VAR2 et BOOLEEN)       ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
Write24C16    CALL      Set_Adr_I2C    ; - Pointer l'adresse
              MOVF      I2C_data,W     ; - Lire l'octet a envoyer (I2C_data)
              CALL      WtoI2C         ; - Envoyer l'octet
              CALL      I2CgetACK      ; - Acquerir l'ACK
              CALL      I2CSTOP        ; - STOP
              BTFSS     BOOLEEN,I2CACK ; - Tester l'ACK :
              GOTO      MemNonDispo1         ; - Si PB : tant pis, abreger !
ProgPasFinie  CALL      I2CSTART             ; - Sinon :
              MOVLW     (I2C_EEPROM+I2C_WR)  ; attendre que la programmation
              CALL      WtoI2C               ; soit terminee en essayant
              CALL      I2CgetACK            ; d'acceder a la memoire.
              CALL      I2CSTOP              ; Quand celle-ci repond (ACK),
              BTFSS     BOOLEEN,I2CACK       ; c'est qu'elle est enfin dispo
              GOTO      ProgPasFinie         ; et qu'elle a fini le boulot.
MemNonDispo1  RETURN                         ; La, c'est fini !

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Dans l'EEPROM I2C, pointe l'adresse memoire voulue :                ;;
;; les 3 bits de poids fort de l'adresse sont mis dans [I2C_adrHIGH].  ;;
;; les 8 bits de poids faible de l'adresse sont mis dans [I2C_adrLOW]. ;;
;; ex pour l'adresse $5BC: $05 dans I2C_adrHIGH et $BC dans I2C_adrLOW ;;
;; (attention, cela modifie les variables VAR1, VAR2 et BOOLEEN)       ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
Set_Adr_I2C   CALL      I2CSTART            ; - 'Start' : debut sequence.
              RLF       I2C_adrHIGH,W       ; - on cree le premier octet :
              ANDLW     00001110b           ;  B3 a B1 : les 3 bits de poids
              IORLW     (I2C_EEPROM+I2C_WR) ;  fort de l'adresse ; B7 a B4 =
              CALL      WtoI2C              ;  [1010] et B0 = 0 (write).
              CALL      I2CgetACK           ; - on verifie si l'octet a bien
              BTFSS     BOOLEEN,I2CACK      ;   ete recu :
              GOTO      MemNonDispo2        ;  .si non, tant pis, on quitte.
              MOVF      I2C_adrLOW,W        ;  .si oui, on envoie l'octet de
              CALL      WtoI2C              ;   poids faible de l'adresse en
              CALL      I2CgetACK           ;   verifiant qu'il est recu.
MemNonDispo2  RETURN                        ;; Si PB : I2CACK=0 dans BOOLEEN

;**************************************************************************;

;--------------------------------------------------------------------------;
;                  sous-fonctions permettant de realiser                   ;
;                 les chronogrammes d'acces a la memoire :                 ;
;--------------------------------------------------------------------------;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Envoie une sequence 'START' au peripherique I2C ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
I2CSTART      BSF       PORTA,SCL ; Au debut, SCL
              BSF       PORTA,SDA ; et SDA sont a '1'.
              CALL      Tempo5us  ; On attend 5 us,
              BCF       PORTA,SDA ; puis SDA passe a '0',
              CALL      Tempo5us  ; puis on attend 5 us,
              BCF       PORTA,SCL ; puis SCL passe a '0',
              CALL      Tempo5us  ; puis on attend 5 us.
              BSF       PORTA,SDA ; SDA doit etre remis a '1' pour pouvoir
              RETURN              ; voir les donnees venant de la memoire.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Envoie une sequence STOP au peripherique I2C ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
I2CSTOP       BCF       PORTA,SCL ; Au debut, SCL
              BCF       PORTA,SDA ; et SDA sont a '0'.
              CALL      Tempo5us  ; On attend 5 us,
              BSF       PORTA,SCL ; puis SCL passe a '1',
              CALL      Tempo5us  ; puis on attend 5 us,
              BSF       PORTA,SDA ; puis SDA passe a '1',
              CALL      Tempo5us  ; on attend 5 us...
              RETURN              ; Fin de la routine.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Envoie un octet contenu dans W au peripherique I2C en 'serie' : ;;
;; (attention, cela modifie les variables VAR1 et VAR2)            ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; L'octet a envoyer est mis dans VAR1 ; on va alors faire subir 8 fois
; a cet octet un decalage vers la gauche, en recopiant a chaque fois au
; prealable le bit B7 sur la broche SDA (avant d'appliquer une impulsion
; d'horloge sur SCL). La variable VAR2 sert de compteur : elle part de 8
; et est decrementee 8 fois jusqu'a 0 afin de compter le nombre de
; decalages :
WtoI2C        MOVWF     VAR1          ; - Octet a envoyer mis dans VAR1.
              MOVLW     8             ; - 8 (nb de decalages) dans VAR2.
              MOVWF     VAR2          ;
NextBitI2Cout BCF       PORTA,SCL     ; - SCL mis a '0'.
              BTFSS     VAR1,7        ; - Bit B7 de VAR1 a '1' ?
              BCF       PORTA,SDA     ;   si non : mettre SDA a '0'.
              BTFSC     VAR1,7        ; - Bit B7 de VAR1 a '0' ?
              BSF       PORTA,SDA     ;   si non : mettre SDA a '1'.
              CALL      Tempo5us      ; - On applique l'impulsion sur SCL :
              BSF       PORTA,SCL     ;   attente 5 us, SCL passe a '1',
              CALL      Tempo5us      ;   attente 5 us, SCL repasse a '0'.
              BCF       PORTA,SCL     ;
              RLF       VAR1,F        ; - On decale VAR1 un coup a gauche.
              DECFSZ    VAR2,F        ; - On decremente VAR2 ; si ce n'est
              GOTO      NextBitI2Cout ;   pas la 8ieme fois, on recommence.
              BSF       PORTA,SDA     ; - On remet SDA a '1' (ce qui permet
              RETURN                  ;   de scruter SDA comme une entree).

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Lit un octet depuis le peripherique I2C et le met dans W : ;;
;; (attention, cela modifie les variables VAR1 et VAR2)       ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Cette routine ressemble beaucoup a la precedente ; mais au lieu
; de recopier le bit B7 sur SDA et de decaler l'octet vers la gauche
; et ce huit fois de suite, on commence par decaler l'octet, puis on
; recopie SDA sur B0, toujours huit fois de suite. Les bits ne sont
; plus retires un a un, mais au contraire empiles les uns apres les
; autres dans VAR1 :
I2CtoW        MOVLW     8            ;-On met 8 dans VAR2, variable compteur
              MOVWF     VAR2         ; qui va comptabiliser les 8 decalages.
              CLRF      VAR1         ;- VAR1 initialisee a 0 (= 00000000b).
NextBitI2Cin  BCF       PORTA,SCL    ;- SCL mis a '0'.
              BSF       PORTA,SDA    ;- SDA mis a '1' pour etre 'lisible'.
              CALL      Tempo5us     ;- attente de 5 us.
              BSF       PORTA,SCL    ;- SCL passe a '1'.
              CALL      Tempo5us     ;- attente de 5 us.
              RLF       VAR1,F       ;- decalage de un coup vers la gauche.
              BCF       VAR1,0       ;- bit B0 initialise a '0'.
              BTFSC     PORTA,SDA    ;- on teste SDA : vaut-il '0' ?
              BSF       VAR1,0       ;  si non (SDA='1'), on met B0 a '1'.
              DECFSZ    VAR2,F       ;- huitieme cycle ?
              GOTO      NextBitI2Cin ;  si non, on reboucle : cycle suivant.
              BCF       PORTA,SCL    ;- apres le dernier cycle : SCL a '0'.
              MOVF      VAR1,W       ;- VAR1 (l'octet lu) est copie dans W.
              RETURN                 ;Fin de la routine

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Verifie si le peripherique I2C renvoie un 'Acknowledge' ; si celui-ci ;;
;; est present, le bit I2CACK est mis a '1', sinon I2CACK est mis a '0'. ;;
;; (la seule variable modifiee est BOOLEEN)                              ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
I2CgetACK     BSF       PORTA,SDA       ; On s'assure que SDA est a '1'.
              BCF       PORTA,SCL       ; On genere l'impulsion positive
              CALL      Tempo5us        ; sur SCL : SCL a '0', attente de 5
              BSF       PORTA,SCL       ; us, SCL a '1', attente de 5 us.
              CALL      Tempo5us        ; La, on va lire la valeur de SDA :
              BCF       BOOLEEN,I2CACK  ; - On met le bit I2CACK a '0'.
              BTFSS     PORTA,SDA       ; - On verifie l'ACK (SDA a '0' ?)
              BSF       BOOLEEN,I2CACK  ;   si oui, OK: on met I2CACK a '1'.
              BCF       PORTA,SCL       ; Puis SCL repasse a '0'.
              RETURN                    ; Et la routine est terminee...

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Envoie un bit 'Acknowledge' au peripherique I2C. ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
I2CsetACK     BCF       PORTA,SCL ; au debut, SCL est a '0'.
              BCF       PORTA,SDA ; SDA est mis a '0' (on applique ACK).
              CALL      Tempo5us  ; on attend 5 us.
              BSF       PORTA,SCL ; Impulsion positive sur SCL :
              CALL      Tempo5us  ; SCL passe a '1' pendant
              BCF       PORTA,SCL ; 5 us, puis repasse a '0'.
              BSF       PORTA,SDA ; Il faut penser a remettre SDA a '1' !
              RETURN ; Fin de la routine, retour au programme principal.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Envoie un 'No Acknowledge' au peripherique I2C. ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
I2CsetNoACK   BCF       PORTA,SCL ; au debut, SCL est a '0'
              BSF       PORTA,SDA ; et SDA a '1' (car pas d'ACK).
              CALL      Tempo5us  ; on attend 5 us.
              BSF       PORTA,SCL ; Impulsion positive sur SCL :
              CALL      Tempo5us  ; SCL passe a '1' pendant
              BCF       PORTA,SCL ; 5 us, puis repasse a '0'.
              RETURN ; Fin de la routine, retour au programme principal.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Temporisation de 5 micro-secondes pour respecter ;;
;; les temps d'acces au peripherique I2C.           ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
Tempo5us      NOP       ; La tempo dure 5 us, avec le "CALL Tempo5us".
              RETURN    ; (CALL=2us) + (NOP=1us) + (RETURN=2us) = 5 us.

              END ; L'assembleur peut maintenant aller faire dodo !

;********************** Fin generale du programme. ***************************;
